<?php
// merged_jobs_feed.php

include_once("dbcommon.php");

// 1. Fetch from manual database source
function fetchFromDatabaseSource($source) {
    $jobs = [];
    $sql = "SELECT * FROM external_jobs 
        WHERE source = " . db_prepare_string($source) . " 
        AND (expiry_date IS NULL OR expiry_date >= CURDATE())
        ORDER BY created_at DESC";
    $rs = DB::Query($sql);

    while ($data = $rs->fetchAssoc()) {
        $jobs[] = [
            'source' => $data['source'],
            'title' => $data['title'],
            'company' => $data['company'],
            'type' => $data['type'],
            'location' => $data['location'],
            'excerpt_preview' => $data['excerpt_preview'],
            'excerpt_full' => $data['excerpt_full'],
            'url' => $data['url'],
			'expiry_date' => $data['expiry_unix']
			
        ];
    }

    return $jobs;
}

// 2. Fetch from Himalayas API (live remote jobs)
function fetchFromHimalayas() {
    $response = file_get_contents("https://himalayas.app/jobs/api");
    $data = json_decode($response, true);
    $jobs = [];
    $now = time();

    foreach ($data['jobs'] ?? [] as $job) {
        $expiry = $job['expiryDate'] ?? null;
        if ($expiry && $expiry < $now) continue;

        $excerpt_full = strip_tags($job['excerpt']);
        $excerpt_preview = mb_substr($excerpt_full, 0, 150);

        $jobs[] = [
            'source' => 'Himalayas',
            'title' => $job['title'],
            'company' => $job['companyName'],
            'type' => $job['employmentType'],
            'location' => implode(', ', $job['locationRestrictions'] ?? []),
            'excerpt_preview' => $excerpt_preview,
            'excerpt_full' => $excerpt_full,
            'url' => $job['applicationLink'] ?? $job['url'],
            'pub_date' => $job['pubDate'] ?? null,
            'expiry_date' => $expiry
        ];
    }

    return $jobs;
}


// ✅ Return grouped jobs by source
function fetchJobsBySource() {
    $jobsBySource = [
        'SPREP' => fetchFromDatabaseSource('SPREP'),
        'SPC' => fetchFromDatabaseSource('SPC'),
        'UNDP' => fetchFromDatabaseSource('UNDP'),
        'Himalayas' => fetchFromHimalayas()
    ];
    return $jobsBySource;
}

// 🔄 Legacy all-in-one fetch
function fetchAllJobs() {
    return array_merge(
        fetchFromDatabaseSource('SPREP'),
        fetchFromDatabaseSource('SPC'),
        fetchFromDatabaseSource('UNDP'),
        fetchFromHimalayas()
    );
}

// 📌 Display disclaimer
function getExternalJobsDisclaimer() {
    return '<div style="text-align:center; max-width:800px; margin: 10px auto 30px; font-size: 14px; color: #333;">
        <em>This feed features job listings curated from publicly available career pages of regional organizations such as <strong>SPREP</strong>, <strong>SPC</strong>, and <strong>UNDP</strong>, alongside global remote opportunities from platforms like <strong>Himalayas</strong>. These links are shared to help jobseekers explore both regional and international roles. All rights, applications, and recruitment processes remain with the respective organizations.</em>
 </div>';
}
