<?php
// merged_jobs_feed.php

include_once("dbcommon.php");

// 🔁 Parse regional RSS feeds (SPC, SPREP, UNDP)
function fetchFromRssFeed($url, $sourceLabel) {
    $jobs = [];
    $rss = @simplexml_load_file($url);
    if (!$rss) return $jobs;

    foreach ($rss->channel->item as $item) {
        $title = (string)$item->title;
        $link = (string)$item->link;
        $description = strip_tags((string)$item->description);
        $excerpt_preview = mb_substr($description, 0, 150);

        $jobs[] = [
            'source' => $sourceLabel,
            'title' => $title,
            'company' => $sourceLabel,
            'type' => 'N/A',
            'location' => 'Pacific Region',
            'excerpt_preview' => $excerpt_preview,
            'excerpt_full' => $description,
            'url' => $link,
            'expiry_date' => strtotime('+30 days') // ✅ default expiry
        ];
    }

    return $jobs;
}

function fetchFromSpcRss() {
    return fetchFromRssFeed('https://www.spc.int/jobs/rss', 'SPC');
}

function fetchFromSprepRss() {
    return fetchFromRssFeed('https://www.sprep.org/job-feed.xml', 'SPREP');
}

function fetchFromUndpRss() {
    return fetchFromRssFeed('https://jobs.undp.org/cj_rss_feed.cfm', 'UNDP');
}

// 2. Fetch from Himalayas API (live remote jobs)
function fetchFromHimalayas() {
    $response = file_get_contents("https://himalayas.app/jobs/api");
    $data = json_decode($response, true);
    $jobs = [];
    $now = time();

    foreach ($data['jobs'] ?? [] as $job) {
        $expiry = $job['expiryDate'] ?? null;
        if ($expiry && $expiry < $now) continue; // skip expired

        $excerpt_full = strip_tags($job['excerpt']);
        $excerpt_preview = mb_substr($excerpt_full, 0, 150);

        $jobs[] = [
            'source' => 'Himalayas',
            'title' => $job['title'],
            'company' => $job['companyName'],
            'type' => $job['employmentType'],
            'location' => implode(', ', $job['locationRestrictions'] ?? []),
            'excerpt_preview' => $excerpt_preview,
            'excerpt_full' => $excerpt_full,
            'url' => $job['applicationLink'] ?? $job['url'],
            'expiry_date' => $expiry
        ];
    }

    return $jobs;
}

// ✅ Return grouped jobs by source
function fetchJobsBySource($limitPerSource = null) {
    $sources = [
        'SPREP' => fetchFromSprepRss(),
        'SPC' => fetchFromSpcRss(),
        'UNDP' => fetchFromUndpRss(),
        'Himalayas' => fetchFromHimalayas()
    ];

    if ($limitPerSource !== null) {
        foreach ($sources as $key => $jobs) {
            $sources[$key] = array_slice($jobs, 0, $limitPerSource);
        }
    }

    return $sources;
}

// 🔄 Legacy all-in-one fetch
function fetchAllJobs() {
    return array_merge(
        fetchFromSprepRss(),
        fetchFromSpcRss(),
        fetchFromUndpRss(),
        fetchFromHimalayas()
    );
}

// 📌 Display disclaimer
function getExternalJobsDisclaimer() {
    return '<div style="text-align:center; max-width:800px; margin: 10px auto 30px; font-size: 14px; color: #333;">
        <em>Test These listings are sourced from publicly available job boards of regional organizations such as SPREP, SPC, and UNDP. We provide these links to help jobseekers access regional opportunities. All rights and applications remain with the respective organizations.</em>
    </div>';
}
