<?php
include_once("dbcommon.php");

function fetchFromDatabaseSource($source) {
    $jobs = [];
    $sourceCondition = $source === 'Ministry'
        ? "source LIKE 'Ministry%'"
        : "source = " . db_prepare_string($source);

    $sql = "
    SELECT * FROM external_jobs
    WHERE $sourceCondition
      AND (
            expiry_date IS NULL 
            OR expiry_date >= CURDATE() - INTERVAL 7 DAY
          )
    ORDER BY title ASC
";

    $rs = DB::Query($sql);
    while ($data = $rs->fetchAssoc()) {
        $jobs[] = [
            'source' => $data['source'],
            'title' => $data['title'],
            'company' => $data['company'],
            'type' => $data['type'],
            'location' => $data['location'],
            'excerpt_preview' => $data['excerpt_preview'],
            'excerpt_full' => $data['excerpt_full'],
            'url' => $data['url'],
            'expiry_date' => $data['expiry_unix']
        ];
    }
    return $jobs;
}

function fetchFromHimalayas() {
    $response = @file_get_contents("https://himalayas.app/jobs/api");
    if (!$response) return [];

    $data = json_decode($response, true);
    $jobs = [];
    $now = time();

    foreach ($data['jobs'] ?? [] as $job) {
        $expiryRaw = $job['expiryDate'] ?? null;

        if (is_numeric($expiryRaw)) {
            $expiry = ($expiryRaw > 9999999999) ? intval($expiryRaw / 1000) : intval($expiryRaw);
        } else {
            $expiry = strtotime($expiryRaw);
        }

        if ($expiry && $expiry < $now) continue;

        $excerpt_full = strip_tags($job['excerpt']);
        $excerpt_preview = mb_substr($excerpt_full, 0, 150);

        $jobs[] = [
            'source' => 'Himalayas',
            'title' => $job['title'],
            'company' => $job['companyName'],
            'type' => $job['employmentType'],
            'location' => implode(', ', $job['locationRestrictions'] ?? []),
            'excerpt_preview' => $excerpt_preview,
            'excerpt_full' => $excerpt_full,
            'url' => $job['applicationLink'] ?? $job['url'],
            'pub_date' => $job['pubDate'] ?? null,
            'expiry_date' => $expiry
        ];
    }

    usort($jobs, fn($a, $b) => strcmp($a['title'], $b['title']));
    return $jobs;
}


function fetchJobsBySource() {
    return [
        'Samoa Government' => fetchFromDatabaseSource('Govt'),
        'Secretariat of the Pacific Regional Environment Programme (SPREP)' => fetchFromDatabaseSource('SPREP'),
        'Pacific Community (SPC)' => fetchFromDatabaseSource('SPC'),
        'United Nations Development Programme (UNDP)' => fetchFromDatabaseSource('UNDP'),
		'Pacific Islands Forum Secretariat (PIFS)' => fetchFromDatabaseSource('PIFS'),
		'Pacific Islands Forum Fisheries Agency (FFA)' => fetchFromDatabaseSource('FFA'),
		'Pacific Vacancies' => fetchFromDatabaseSource('PJN'),
		'User-Posted' => fetchFromDatabaseSource('Public'),
        'Himalayas (Live API Feed)' => fetchFromHimalayas()
    ];
}

function fetchAllJobs() {
    return array_merge(
        fetchFromDatabaseSource('Govt'),
        fetchFromDatabaseSource('SPREP'),
        fetchFromDatabaseSource('SPC'),
        fetchFromDatabaseSource('UNDP'),
		fetchFromDatabaseSource('PIFS'),
		fetchFromDatabaseSource('FFA'),
		fetchFromDatabaseSource('PJN'),
		fetchFromDatabaseSource('Public'),
        fetchFromHimalayas()
    );
}

function getExternalJobsDisclaimer() {
    return '<div style="text-align:center; max-width:800px; margin: 10px auto 30px; font-size: 14px; color: #333;">
        <em>This live feed showcases job opportunities gathered from publicly accessible career pages of regional organizations such as SPREP, SPC, FFA, PIFS, and UNDP, as well as government ministry vacancies listed through the Samoa Public Service Commission (PSC), Ministries & Corporations. It also includes global remote job listings from platforms like Himalayas.</em><br><br>
        <em>These listings are shared to assist jobseekers in discovering a wide range of local, regional, and international employment opportunities. Please note that all application procedures, rights, and recruitment decisions remain solely with the respective hiring organizations or agencies.</em>
    </div>';
}
