<?php
include_once("dbcommon.php");

/**
 * Fetch jobs from the local external_jobs table by source, filtered by search term
 */
function fetchFromDatabaseSource($source, $searchTerm = '') {
    $jobs = [];
    $sourceCondition = $source === 'Ministry'
        ? "source LIKE 'Ministry%'"
        : "source = " . db_prepare_string($source);

/*
if (!empty($searchTerm)) {
    $escaped = db_prepare_string("%$searchTerm%");
    $searchCondition = " AND (
        title LIKE $escaped
        OR location LIKE $escaped
        OR type LIKE $escaped
        OR company LIKE $escaped
    )";
}*/


    $sql = "
    SELECT * FROM external_jobs
    WHERE $sourceCondition
      AND (
            expiry_date IS NULL 
            OR expiry_date >= CURDATE() - INTERVAL 7 DAY
          )
      $searchCondition
    ORDER BY title ASC
    ";

    $rs = DB::Query($sql);
    while ($data = $rs->fetchAssoc()) {
        $jobs[] = [
            'source' => $data['source'],
            'title' => $data['title'],
            'company' => $data['company'],
            'type' => $data['type'],
            'location' => $data['location'],
            'excerpt_preview' => $data['excerpt_preview'],
            'excerpt_full' => $data['excerpt_full'],
            'url' => $data['url'],
//          'expiry_date' => $data['expiry_unix']
			'expiry_date' => !empty($data['expiry_unix']) ? $data['expiry_unix'] : strtotime($data['expiry_date'] ?? 'now')
        ];
    }
    return $jobs;
}

/**
 * Fetch jobs from Himalayas API, filtered by search term
 */
function fetchFromHimalayas($searchTerm = '') {
    $response = @file_get_contents("https://himalayas.app/jobs/api");
    if (!$response) return [];

    $data = json_decode($response, true);
    $jobs = [];
    $now = time();

    foreach ($data['jobs'] ?? [] as $job) {
        $expiryRaw = $job['expiryDate'] ?? null;

        if (is_numeric($expiryRaw)) {
            $expiry = ($expiryRaw > 9999999999) ? intval($expiryRaw / 1000) : intval($expiryRaw);
        } else {
            $expiry = strtotime($expiryRaw);
        }

        if ($expiry && $expiry < $now) continue;

        if (!empty($searchTerm) && stripos($job['title'], $searchTerm) === false) {
            continue; // Skip if title does not match
        }

        $excerpt_full = strip_tags($job['excerpt']);
        $excerpt_preview = mb_substr($excerpt_full, 0, 150);

        $jobs[] = [
            'source' => 'Himalayas',
            'title' => $job['title'],
            'company' => $job['companyName'],
            'type' => $job['employmentType'],
            'location' => implode(', ', $job['locationRestrictions'] ?? []),
            'excerpt_preview' => $excerpt_preview,
            'excerpt_full' => $excerpt_full,
            'url' => $job['applicationLink'] ?? $job['url'],
            'pub_date' => $job['pubDate'] ?? null,
            'expiry_date' => $expiry
        ];
    }

    usort($jobs, fn($a, $b) => strcmp($a['title'], $b['title']));
    return $jobs;
}

/**
 * Return jobs grouped by their source
 */
function fetchJobsBySource($searchTerm = '') {
    return [
        'Samoa Government' => fetchFromDatabaseSource('Govt', $searchTerm),
        'Secretariat of the Pacific Regional Environment Programme (SPREP)' => fetchFromDatabaseSource('SPREP', $searchTerm),
        'Pacific Community (SPC)' => fetchFromDatabaseSource('SPC', $searchTerm),
        'United Nations Development Programme (UNDP)' => fetchFromDatabaseSource('UNDP', $searchTerm),
        'Pacific Islands Forum Secretariat (PIFS)' => fetchFromDatabaseSource('PIFS', $searchTerm),
        'Pacific Islands Forum Fisheries Agency (FFA)' => fetchFromDatabaseSource('FFA', $searchTerm),
        'Pacific Vacancies' => fetchFromDatabaseSource('PJN', $searchTerm),
        'User-Posted' => fetchFromDatabaseSource('Public', $searchTerm),
        'Himalayas (Live API Feed)' => fetchFromHimalayas($searchTerm)
    ];
}

/**
 
 * Convert long or irregular source labels to CSS-safe slugs
 */
function getSourceSlug($sourceLabel) {
    $map = [
        'Samoa Government' => 'govt',
        'Secretariat of the Pacific Regional Environment Programme (SPREP)' => 'sprep',
        'Pacific Community (SPC)' => 'spc',
        'United Nations Development Programme (UNDP)' => 'undp',
        'Pacific Islands Forum Secretariat (PIFS)' => 'pifs',
        'Pacific Islands Forum Fisheries Agency (FFA)' => 'ffa',
        'Pacific Vacancies' => 'pjn',
        'User-Posted' => 'public',
        'Himalayas (Live API Feed)' => 'himalayas'
    ];

    return isset($map[$sourceLabel])
        ? $map[$sourceLabel]
        : strtolower(preg_replace('/[^a-z0-9]+/', '-', $sourceLabel));
}




/**
 * Return a flat list of all jobs (merged from all sources)
 */
function fetchAllJobs($searchTerm = '') {
    return array_merge(
        fetchFromDatabaseSource('Govt', $searchTerm),
        fetchFromDatabaseSource('SPREP', $searchTerm),
        fetchFromDatabaseSource('SPC', $searchTerm),
        fetchFromDatabaseSource('UNDP', $searchTerm),
        fetchFromDatabaseSource('PIFS', $searchTerm),
        fetchFromDatabaseSource('FFA', $searchTerm),
        fetchFromDatabaseSource('PJN', $searchTerm),
        fetchFromDatabaseSource('Public', $searchTerm),
        fetchFromHimalayas($searchTerm)
    );
}

/**
 * Job disclaimer for the bottom of the merged feed
 */
function getExternalJobsDisclaimer() {
    return '<div style="text-align:center; max-width:800px; margin: 10px auto 30px; font-size: 14px; color: #333;">
        <em>This live feed showcases job opportunities gathered from publicly accessible career pages of regional organizations such as SPREP, SPC, FFA, PIFS, and UNDP, as well as government ministry vacancies listed through the Samoa Public Service Commission (PSC), Ministries & Corporations. It also includes global remote job listings from platforms like Himalayas.</em><br><br>
        <em>These listings are shared to assist jobseekers in discovering a wide range of local, regional, and international employment opportunities. Please note that all application procedures, rights, and recruitment decisions remain solely with the respective hiring organizations or agencies.</em>
    </div>';
}
